implementation module Highscore


/*
	General utility for reading/writing high scores to Files and displaying current high scores.
	This module uses the 0.8 I/O library.
*/

import	StdBool, StdString, StdFile, StdEnum, StdTuple, StdList, StdMisc
from	deltaSystem	import HomePath
import	deltaIOSystem, deltaDialog

::	*HiScores	:== (!Files,!Highs)
::	Highs		:== [High]
::	High
	=	{	name	:: !String
		,	score	:: !Int
		}

//	Read in the high scores:
ReadHiScores :: !String !Files -> (!*File,!HiScores)
ReadHiScores fname files
#	(exists,file,files)	= fopen fpath FReadData files
|	exists				= (file1, (files,highs))
						with
							(highs,file1)	= ReadHighs file
							
							ReadHighs :: !*File -> (!Highs,!*File)
							ReadHighs file
							|	sfend file	= ([],file)
							#	(name, file)= freads file 13
								(ok,hi,file)= freadi file
							|	not ok		= ([],file)
							#	(ok,_, file)= freadc file
							|	not ok		= ([],file)
							#	(rest, file)= ReadHighs file
							|	otherwise	= ([{name=name,score=hi}:rest],file)
#	(_,create,files)	= fopen fpath FWriteData files
|	otherwise			= (create,(files,[]))
where
	fpath				= HomePath fname

//	Write the high scores:
WriteHiScores :: !*File !HiScores -> Files
WriteHiScores file (files,highs)
#	(ok,file)	= freopen file FWriteData
|	not ok		= abort "Could not reopen file.\n"
#	file		= WriteHighs highs file
	(_,files)	= fclose file files
=	files
where
	WriteHighs :: !Highs !*File -> *File
	WriteHighs [{name,score}:scores] file
	#	file	= fwrites  (ThirteenCharString name) file
		file	= fwritei  score					 file
		file	= fwritec '\n'						 file
		file	= WriteHighs scores					 file
	=	file
	WriteHighs _ file = file
	
	ThirteenCharString :: !String -> String
	ThirteenCharString string = (string+++"             ")%(0,12)


//	Determine whether, given the number of high scores, a given score is actually a new high score:
ItsAHighScore :: !Int !Int !Highs -> Bool
ItsAHighScore nrOfHiScores score scores
|	score==0					= False
|	length scores<nrOfHiScores	= True
|	otherwise					= IsItReallyAHighScore score scores
where
	IsItReallyAHighScore :: !Int !Highs -> Bool
	IsItReallyAHighScore score` [{score}:hiscores]
	|	score`>score			= True
	|	otherwise				= IsItReallyAHighScore score` hiscores
	IsItReallyAHighScore _ _	= False


//	Add a High to the current list of high scores:
AddScore :: !Int !High !Highs -> Highs
AddScore nrOfHighScores hi hiscores
=	take nrOfHighScores (addscore hi hiscores)
where
	addscore :: !High !Highs -> Highs
	addscore hi` hiscores=:[hi:his]
	|	hi.score>hi`.score	= [hi  : addscore hi` his]
	|	otherwise			= [hi` : hiscores]
	addscore hi` _			= [hi`]


//	Display high scores in a modal dialog to the user:
ShowHiScores :: DialogId String !Highs !*s !(IOState *s) -> (!*s,!IOState *s)
ShowHiScores _ _ [] state io
#	(_,state,io)	= OpenNotice (Notice ["No high scores available."] (NoticeButton 1 "OK") []) state io
=	(state,io)
ShowHiScores id header highs state io
=	OpenModalDialog dialog state io
where
	dialog	= CommandDialog id "High Scores" [] okId
				[	StaticText 1 Center header
				:	flatten
				[	
				[	DynamicText id		(YOffset (id-2) (Pixel 2)) (MM 65.0) (toString ((id-2)/2)+++". "+++name)
				,	StaticText  (id+1)	(XOffset id (MM 0.0)) (toString score)
				]	\\ (id,{name,score}) <- zip2 [4,6..] highs
				]
				++	
				[	DialogButton okId Center "OK" Able (\_ state io -> (state, CloseActiveDialog io))
				]
				]
	okId	= 2*(length highs+1)
